// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import UIKit
import MSSNotificationClient

@UIApplicationMain
class AppDelegate: UIResponder, UIApplicationDelegate {
    var window: UIWindow?
    
    var notificationFromBackground = false
    var errorText: String?
    var notificationIdentifierText: String?
    var oneSpanNotification: OneSpanNotification?

    func application(_ application: UIApplication, didFinishLaunchingWithOptions launchOptions: [UIApplication.LaunchOptionsKey: Any]?) -> Bool {
        print("Version: \(NotificationClientSDK.version)")
        // We want subscribe to every availlable aspect of the notification
        let notificationTypes: NotificationType = [.badge, .sound, .alert]
        
        do {
            try NotificationClientSDK.registerNotificationServices(withType: notificationTypes)
        } catch let error {
            handleNotificationSDKError(error: error)
        }
        
        return true
    }
    
    func application(_ application: UIApplication, didRegisterForRemoteNotificationsWithDeviceToken deviceToken: Data) {
        do {
            // Create the OneSpan Notification Identifier from the token provided by the system
            notificationIdentifierText = try NotificationClientSDK.oneSpanNotificationIdentifier(for: deviceToken)
            print("OneSpan notification ID: \(notificationIdentifierText ?? "nil")")
            notify()
        } catch let error {
            notificationIdentifierText = nil
            handleNotificationSDKError(error: error)
        }
    }
    
    func application(_ application: UIApplication, didFailToRegisterForRemoteNotificationsWithError error: Error) {
        print("Error during registration. Error: \(error.localizedDescription)")
        errorText = error.localizedDescription
        notify()
    }
    
    func application(_ application: UIApplication, didReceiveRemoteNotification userInfo: [AnyHashable: Any], fetchCompletionHandler completionHandler: @escaping (UIBackgroundFetchResult) -> Void) {
        print("didReceiveRemoteNotification userInfo: [AnyHashable : Any], fetchCompletionHandler: \(userInfo.description)")
        
        if application.applicationState == .inactive || application.applicationState == .background {
            // opened from a push notification when the app was on background
            notificationFromBackground = true
        }
        
        // Check that the notification is a correct OneSpan notification
        if NotificationClientSDK.isOneSpanNotification(userInfo) {
            manageNotificationWithApplication(application, userInfo: userInfo)
        }
        
        completionHandler(.newData)
    }

    func applicationWillResignActive(_ application: UIApplication) {
        // Sent when the application is about to move from active to inactive state. This can occur for certain types of temporary interruptions (such as an incoming phone call or SMS message) or when the user quits the application and it begins the transition to the background state.
        // Use this method to pause ongoing tasks, disable timers, and invalidate graphics rendering callbacks. Games should use this method to pause the game.
    }

    func applicationDidEnterBackground(_ application: UIApplication) {
        // Use this method to release shared resources, save user data, invalidate timers, and store enough application state information to restore your application to its current state in case it is terminated later.
        // If your application supports background execution, this method is called instead of applicationWillTerminate: when the user quits.
    }

    func applicationWillEnterForeground(_ application: UIApplication) {
        // Called as part of the transition from the background to the active state; here you can undo many of the changes made on entering the background.
    }

    func applicationDidBecomeActive(_ application: UIApplication) {
        // Restart any tasks that were paused (or not yet started) while the application was inactive. If the application was previously in the background, optionally refresh the user interface.
    }

    func applicationWillTerminate(_ application: UIApplication) {
        // Called when the application is about to terminate. Save data if appropriate. See also applicationDidEnterBackground:.
    }

    private func notify() {
        NotificationCenter.default.post(name: Notification.Name.notificationDataUpdated, object: nil)
    }
    
    private func notifyNotificationReceived() {
        NotificationCenter.default.post(name: Notification.Name.notificationReceived, object: nil)
    }
    
    /**
     * Parse an incomming OneSpan Notification and update the text data accordingly so that the attached
     * ViewController can display them/
     */
    private func manageNotificationWithApplication(_ application: UIApplication, userInfo: [AnyHashable: Any]) {
        UIApplication.shared.applicationIconBadgeNumber = 0

        oneSpanNotification = NotificationClientSDK.oneSpanNotification(from: userInfo)
        
        if oneSpanNotification != nil {
            notifyNotificationReceived()
        }
    }

    private func handleNotificationSDKError(error: Error) {
        if let notificationError = error as? NotificationError {
            errorText = [
                "errorCode: \(notificationError.errorCode)",
                notificationError.errorDescription,
                notificationError.failureReason,
                notificationError.recoverySuggestion
            ]
            .compactMap { $0 }
            .filter { !$0.isEmpty }
            .joined(separator: "\n")
        } else {
            errorText = error.localizedDescription
        }
        
        notify()
    }
}
